#include <stdio.h>
#include <stdlib.h>
#include <sys/random.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include <seccomp.h>

static void sandbox(void) {
    int rc = -1;
    scmp_filter_ctx ctx;
    ;
    if (
        (ctx = seccomp_init(SCMP_ACT_ERRNO(ENOSYS))) == NULL ||
        seccomp_rule_add(ctx, SCMP_ACT_ALLOW, SCMP_SYS(open), 0) < 0 ||
        seccomp_rule_add(ctx, SCMP_ACT_ALLOW, SCMP_SYS(openat), 0) < 0 ||
        seccomp_rule_add(ctx, SCMP_ACT_ALLOW, SCMP_SYS(close), 0) < 0 ||
        seccomp_rule_add(ctx, SCMP_ACT_ALLOW, SCMP_SYS(read), 0) < 0 ||
        seccomp_rule_add(ctx, SCMP_ACT_ALLOW, SCMP_SYS(write), 0) < 0 ||
        seccomp_rule_add(ctx, SCMP_ACT_ALLOW, SCMP_SYS(close), 0) < 0 ||
        seccomp_rule_add(ctx, SCMP_ACT_ALLOW, SCMP_SYS(exit_group), 0) < 0 ||
        seccomp_rule_add(ctx, SCMP_ACT_ALLOW, SCMP_SYS(exit), 0) < 0 ||
        seccomp_load(ctx) < 0)
    {
        _exit(1);
    }
    seccomp_release(ctx);
}

struct files_t {
    int *ptr;
    size_t n;
};

static struct files_t open_random_files(void) {
    const int maximum = 256;

    unsigned n = 128;
    getrandom(&n, sizeof(n), 0);
    n %= maximum;

    struct files_t res;

    res.ptr = malloc(sizeof(int) * n);
    res.n = n;

    for (unsigned i = 0; i < n; i++) {
        int fd = open("/dev/null", 0);
        if (fd < 0) {
            dprintf(STDERR_FILENO, "failed to open file: %s\n", strerror(errno));
        }
        res.ptr[i] = fd;
    }

    return res;
}

int main(int argc, char **argv) {
    setvbuf(stdout, NULL, _IONBF, 0);

    const char *filename = "/etc/passwd";
    if (argc > 1) {
        filename = argv[1];
    }

    open_random_files();

    sandbox();

    printf("from this point on, only open, close, read, and write and exit syscalls are available\n");
    printf("use them to read %s\n", filename);
    printf("here are some useful numbers:\n");
    printf("open - %p\n", open);
    printf("close - %p\n", close);
    printf("read - %p\n", read);
    printf("write - %p\n", write);
    printf("exit - %p\n", exit);
    printf("main - %p\n", main);
    printf("filename - %p\n", filename);
    printf("open_random_files - %p\n", open_random_files);
    printf("\n");

    uintptr_t *p = __builtin_frame_address(0) + sizeof(void*);
    for (;;) {
        char buf[32];
        if (fgets(buf, sizeof(buf), stdin) == 0) {
            exit(1);
        }
        if (buf[0] == '\n' || buf[0] == '\0') break;
        char *endptr;
        long res = strtol(buf, &endptr, 16);
        if (buf == endptr) {
            exit(1);
        }
        *p++ = res;
    }
}
